<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Cliente HTTP para a API do CryptNote.
 */
class CryptNote_Client
{
    private string $api_base;
    private string $api_key;

    public function __construct(array $settings = [])
    {
        $base = $settings['api_base'] ?? 'https://cryptnote.pro/api/v1';
        $this->api_base = rtrim((string)$base, '/');
        $this->api_key = (string)($settings['api_key'] ?? '');
    }

    /**
     * Cria um link criptografado via API.
     *
     * @param array $payload
     * @return array|WP_Error
     */
    public function create(array $payload)
    {
        $content = (string)($payload['content'] ?? '');
        if (trim($content) === '') {
            return new WP_Error('cryptnote_empty_content', __('O conteúdo é obrigatório.', 'cryptnote'));
        }

        $url = $this->api_base . '/create.php';
        $headers = ['Content-Type' => 'application/json'];
        if ($this->api_key !== '') {
            $headers['X-API-Key'] = $this->api_key;
        }

        $response = wp_remote_post($url, [
            'headers' => $headers,
            'body' => wp_json_encode($payload),
            'timeout' => 15,
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $status = (int)wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $json = json_decode($body, true);

        if (!is_array($json)) {
            return new WP_Error('cryptnote_invalid_response', __('Resposta inválida da API CryptNote.', 'cryptnote'), ['status' => $status]);
        }

        if ($status >= 200 && $status < 300 && !empty($json['success'])) {
            return $json;
        }

        $message = $json['error'] ?? __('Erro desconhecido ao comunicar com o CryptNote.', 'cryptnote');
        return new WP_Error('cryptnote_api_error', $message, ['status' => $status, 'response' => $json]);
    }
}
