<?php
if (!defined('ABSPATH')) {
    exit;
}

class CryptNote_Email
{
    private static bool $processing = false;

    public function __construct()
    {
        add_filter('wp_mail', [$this, 'filter_mail'], 5);
    }

    /**
     * Substitui o corpo do e-mail por um link seguro quando habilitado.
     *
     * @param array $args
     * @return array
     */
    public function filter_mail(array $args): array
    {
        $settings = cryptnote_pro_get_settings();
        if (empty($settings['email_enable']) || self::$processing) {
            return $args;
        }

        self::$processing = true;

        $format = $settings['default_format'] ?? (!empty($settings['default_markdown']) ? 'markdown' : 'text');
        if (!in_array($format, ['text', 'markdown', 'html'], true)) {
            $format = 'text';
        }
        $isMarkdown = $format === 'markdown';
        $isHtml = $format === 'html';

        $payload = [
            'content' => $args['message'] ?? '',
            'max_views' => (int)($settings['default_max_views'] ?? 1),
            'is_markdown' => $isMarkdown,
            'is_html' => $isHtml,
        ];

        if ($payload['max_views'] < 1 || $payload['max_views'] > 20) {
            $payload['max_views'] = 1;
        }

        if ($settings['default_expire_minutes'] !== '') {
            $payload['expire_minutes'] = (int)$settings['default_expire_minutes'];
        }

        $client = new CryptNote_Client($settings);
        $result = $client->create($payload);

        if (is_wp_error($result)) {
            $args['message'] = sprintf(
                /* translators: %s: error message */
                __('Não foi possível criar o link seguro (%s). O conteúdo original não foi enviado.', 'cryptnote'),
                $result->get_error_message()
            );
            $args['attachments'] = [];
        } else {
            $share_url = esc_url_raw((string)($result['share_url'] ?? ''));
            $token = sanitize_text_field((string)($result['token'] ?? ''));
            $template = $settings['email_template'] ?: cryptnote_pro_default_settings()['email_template'];
            $templateHtml = $settings['email_template_html'] ?: cryptnote_pro_default_settings()['email_template_html'];
            $expire = $payload['expire_minutes'] ?? '';
            $expireLabel = ($expire === '' ? __('sem expiração definida', 'cryptnote') : $expire);

            $replacements = [
                '{share_url}' => $share_url,
                '{share_link}' => '<a href="' . esc_url($share_url) . '" target="_blank" rel="noopener noreferrer">' . esc_html($share_url) . '</a>',
                '{token}' => $token,
                '{max_views}' => $payload['max_views'],
                '{expire_minutes}' => $expireLabel,
            ];

            if (!empty($settings['email_send_html'])) {
                $args['message'] = $this->format_html_email($templateHtml, $replacements);
                $args['headers'] = $this->force_html_header($args['headers'] ?? []);
            } else {
                $args['message'] = strtr($template, $replacements);
                $args['headers'] = $this->force_plain_text_header($args['headers'] ?? []);
            }
            $args['attachments'] = [];
        }

        self::$processing = false;
        return $args;
    }

    /**
     * Garante que o Content-Type seja texto simples para evitar HTML não criptografado.
     *
     * @param array|string $headers
     * @return array
     */
    private function force_plain_text_header($headers): array
    {
        if (empty($headers)) {
            return ['Content-Type: text/plain; charset=UTF-8'];
        }

        if (is_string($headers)) {
            $headers = preg_split('/\r?\n/', $headers);
        }

        $headers = array_filter(array_map('trim', (array)$headers));
        $has_content_type = false;

        foreach ($headers as $header) {
            if (stripos($header, 'content-type:') === 0) {
                $has_content_type = true;
                break;
            }
        }

        if (!$has_content_type) {
            $headers[] = 'Content-Type: text/plain; charset=UTF-8';
        }

        return $headers;
    }

    /**
     * Define Content-Type como HTML para permitir links clicáveis.
     *
     * @param array|string $headers
     * @return array
     */
    private function force_html_header($headers): array
    {
        if (empty($headers)) {
            return ['Content-Type: text/html; charset=UTF-8'];
        }

        if (is_string($headers)) {
            $headers = preg_split('/\r?\n/', $headers);
        }

        $headers = array_filter(array_map('trim', (array)$headers));
        $contentTypeSet = false;

        foreach ($headers as $index => $header) {
            if (stripos($header, 'content-type:') === 0) {
                $headers[$index] = 'Content-Type: text/html; charset=UTF-8';
                $contentTypeSet = true;
                break;
            }
        }

        if (!$contentTypeSet) {
            $headers[] = 'Content-Type: text/html; charset=UTF-8';
        }

        return $headers;
    }

    /**
     * Constrói o corpo HTML substituindo placeholders e garantindo link clicável.
     */
    private function format_html_email(string $template, array $replacements): string
    {
        $body = strtr($template, $replacements);
        if (strpos($body, '<') === false) {
            return wpautop($body);
        }
        return $body;
    }
}
